package furny.ga.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.commons.lang3.ObjectUtils;

import furny.entities.Furniture;
import furny.entities.Tag;
import furny.furndb.FurnCache;
import furny.ga.FurnEntry;
import furny.ga.FurnLayoutIndividual;
import ga.core.algorithm.util.RandomSingleton;

/**
 * Utility class for handling furniture similarities.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public final class FurnitureUtil {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(FurnitureUtil.class
      .getName());

  /**
   * This map caches all distance calculations, so they are reusable.
   */
  private static final Map<FurniturePair, Double> SIMILARITY_MAP = new HashMap<FurnitureUtil.FurniturePair, Double>();

  /**
   * Instantiation is not allowed.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private FurnitureUtil() {
  }

  /**
   * Calculates the similarity between two furnitures by their tags. This method
   * caches the results.
   * 
   * @param furn1
   *          The first furniture.
   * @param furn2
   *          The second furniture.
   * @return The similarity in [0;1].
   * 
   * @see #getSimilarityImpl(Furniture, Furniture)
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static double getSimilarity(final Furniture furn1,
      final Furniture furn2) {
    if (furn1.equals(furn2)) {
      return 1d;
    }

    final FurniturePair key = new FurniturePair(furn1, furn2);

    Double d = SIMILARITY_MAP.get(key);
    if (d == null) {
      d = getSimilarityImpl(furn1, furn2);
      SIMILARITY_MAP.put(key, d);
    }

    return d;
  }

  /**
   * Calculates the similarity between two furnitures by their tags.
   * 
   * @param furn1
   *          The first furniture.
   * @param furn2
   *          The second furniture.
   * @return The similarity in [0;1].
   * 
   * @see #getSimilarity(Furniture, Furniture)
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private static double getSimilarityImpl(final Furniture furn1,
      final Furniture furn2) {
    final double dScal = 1 - getManhattanDistance(furn1.getMetaData()
        .getWidth(), furn2.getMetaData().getWidth(), furn1.getMetaData()
        .getLength(), furn2.getMetaData().getLength());
    final double dTag = getTagSimilarity(furn1, furn2);

    if (LOGGER.isLoggable(Level.FINE)) {
      LOGGER.fine("similarity scal " + dScal);
      LOGGER.fine("similarity tag " + dTag);
      LOGGER.fine("similarity total " + (dScal + dTag) / 2d);
    }
    return (dScal + dTag) / 2d;
  }

  /**
   * Calculates the manhattan distance of width and length.
   * 
   * @param w1
   *          Width 1
   * @param w2
   *          Width 2
   * @param l1
   *          Length 1
   * @param l2
   *          Length 2
   * @return Manhattan distance.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private static double getManhattanDistance(final double w1, final double w2,
      final double l1, final double l2) {
    if (LOGGER.isLoggable(Level.FINE)) {
      LOGGER.fine(w1 + ", " + w2 + ", " + l1 + ", " + l2 + ",");
    }
    return (Math.abs(w1 - w2) + Math.abs(l1 - l2))
        / (Math.max(w1, w2) + Math.max(l1, l2));
  }

  /**
   * Calculates the similarity between two sets of tags.
   * 
   * @param furn1
   *          The first furniture.
   * @param furn2
   *          The second furniture.
   * @return The similarity in [0;1].
   * 
   * @see #getSimilarity(Furniture, Furniture)
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private static double getTagSimilarity(final Furniture furn1,
      final Furniture furn2) {
    final Set<Tag> tags1 = furn1.getMetaData().getTags();
    final Set<Tag> tags2 = furn2.getMetaData().getTags();

    double ranking1 = 0d;
    double ranking2 = 0d;

    double matches = 0d;

    for (final Tag t1 : tags1) {
      ranking1 += t1.getType().getRanking();

      if (tags2.contains(t1)) {
        matches += t1.getType().getRanking();
      }
    }

    for (final Tag t2 : tags2) {
      ranking2 += t2.getType().getRanking();
    }

    if (ranking1 > 0d || ranking1 > 0d) {
      return matches / Math.max(ranking1, ranking2);
    } else {
      return 0d;
    }
  }

  /**
   * Creates a sorted map of furnitures and their similarities to the given
   * furniture.
   * 
   * @param furn1
   *          The furniture to calculate similarities for.
   * @param allFurnitures
   *          All other furnitures.
   * @return Map of Furnitures and similarities sorted by their similarities.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static Map<Furniture, Double> getSimilarityRankMap(
      final Furniture furn1, final List<Furniture> allFurnitures) {

    final Map<Furniture, Double> all = new HashMap<Furniture, Double>();

    for (final Furniture f : allFurnitures) {
      all.put(f, getSimilarity(furn1, f));
    }

    return createSortedMap(all);
  }

  /**
   * Creates a descending sorted map from the given map.
   * 
   * @param passedMap
   *          The map to create the sorted map from.
   * @return The sorted map.
   * 
   * @param <T>
   *          Generic type parameter
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private static <T extends Comparable<T>> Map<Furniture, T> createSortedMap(
      final Map<Furniture, T> passedMap) {
    final List<T> mapValues = new ArrayList<T>(passedMap.values());
    Collections.sort(mapValues, Collections.reverseOrder());

    final Map<Furniture, T> sortedMap = new LinkedHashMap<Furniture, T>();

    final Iterator<T> kIt = mapValues.iterator();
    while (kIt.hasNext()) {
      final T value = kIt.next();

      for (final Entry<Furniture, T> entry : passedMap.entrySet()) {
        if (entry.getValue() == value) {
          sortedMap.put(entry.getKey(), value);
        }
      }
    }
    return sortedMap;
  }

  /**
   * This method does a roulette selection on the sorted map of similarities
   * (rank map) to find a second furniture.
   * 
   * @param f1
   *          Furniture
   * @return A furniture with a high similarity to the first one.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static Furniture getOtherFurnitureBySimilarityRoulette(
      final Furniture f1) {
    return rouletteImpl(getSimilarityRankMap(f1, FurnCache.getInstance()
        .getAllFurnitures()));
  }

  /**
   * Gets the furniture with a distance inside of the given width.
   * 
   * @param f1
   *          The furniture.
   * @param dist
   *          The width where the other furniture should be inside.
   * @return The other furniture.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static Furniture getOtherFurnitureBySimilarityDistance(
      final Furniture f1, final int dist) {
    final Map<Furniture, Double> map = getSimilarityRankMap(f1, FurnCache
        .getInstance().getAllFurnitures());

    final List<Furniture> list = new ArrayList<Furniture>(map.keySet());
    Collections.sort(list, new Comparator<Furniture>() {
      @Override
      public int compare(final Furniture o1, final Furniture o2) {
        final double n1 = map.get(o1);
        final double n2 = map.get(o2);

        if (n1 < n2) {
          return 1;
        } else if (n1 > n2) {
          return -1;
        } else {
          return 0;
        }
      }
    });

    return list.get(Math.min(dist, list.size() - 1));
  }

  /**
   * Roulette algorithm for maps. Adapted from <a
   * href="http://stackoverflow.com/a/1575995/748524">Stack overflow</a>.
   * 
   * @param map
   *          The map of furnitures and similarities.
   * @return The selected furniture.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private static Furniture rouletteImpl(final Map<Furniture, Double> map) {
    final Furniture[] furns = new Furniture[map.size()];

    double totalValue = 0;
    int i = 0;
    for (final Map.Entry<Furniture, Double> entry : map.entrySet()) {
      furns[i++] = entry.getKey();
      totalValue += entry.getValue();
    }

    double randNum = RandomSingleton.getRandom().nextDouble() * totalValue;
    int idx;
    for (idx = 0; idx < furns.length && randNum > 0; ++idx) {
      randNum -= map.get(furns[idx]);
    }
    return furns[idx - 1];
  }

  /**
   * Calculates the total costs of a furniture layout.
   * 
   * @param ind
   *          The individual.
   * @return The total cost.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static float getCosts(final FurnLayoutIndividual ind) {
    double costs = 0f;

    // add as double, than convert to float. so not much precision should be
    // lost
    for (final FurnEntry entry : ind.getFurnitures()) {
      costs += entry.getFurniture().getMetaData().getPrice();
    }
    return (float) costs;
  }

  /**
   * Sorted pair for caching similarities.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private static class FurniturePair {
    private final Furniture smaller, bigger;

    /**
     * Creates a pair.
     * 
     * @param f1
     *          Furniture one.
     * @param f2
     *          Furniture two.
     * 
     * @since 11.08.2012
     * @author Stephan Dreyer
     */
    public FurniturePair(final Furniture f1, final Furniture f2) {
      // the smaller id
      this.smaller = f1.getId() < f2.getId() ? f1 : f2;

      // the other id
      this.bigger = smaller == f1 ? f2 : f1;
    }

    @Override
    public int hashCode() {
      return ObjectUtils.hashCodeMulti(smaller, bigger);
    }

    @Override
    public boolean equals(final Object obj) {
      return obj != null && obj.getClass() == getClass()
          && ((FurniturePair) obj).smaller.equals(smaller)
          && ((FurniturePair) obj).bigger.equals(bigger);
    }
  }

  /**
   * Main method for test purposes.
   * 
   * @param args
   *          No arguments.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static void main(final String[] args) {
    final Furniture f1 = FurnCache.getInstance().getAllFurnitures().get(0);
    final Furniture f2 = FurnCache.getInstance().getAllFurnitures().get(1);

    System.out.println(f1.getName());
    System.out.println(f2.getName());

    System.out.println(getSimilarity(f1, f2));
    System.out.println(getSimilarity(f2, f1));
    System.out.println(getSimilarity(f1, f2));
    System.out.println(getSimilarity(f2, f2));
    System.out.println(getSimilarity(f1, f1));

  }
}
